//
//  Chap8.h
//  Sinc
//
//  Created by 杉浦 洋 on 2018/04/02.
//  Copyright © 2018年 杉浦 洋. All rights reserved.
//

#ifndef Chap8_h
#define Chap8_h
/*----------------------------------*/
/*［８］階層確率(Level Probability)　*/
/*----------------------------------*/
//Level Probabiliry with k<=4 by Robertson et. al.
double rho(double w1,double w2,double w3){
    return asin(sqrt(w1*w3/((w1+w2)*(w2+w3))));
}
double LvPrLe4(int L,int k,double *w){/*=P(L,k,w), 1≤L≤k≤4 */
    double A,B,C;
    if(k==1)return 1;
    if(k==2)return 0.5;
    if(k==3){
        if(L==2)return 0.5;
        A=rho(w[0],w[1],w[2])/(2*M_PI);
        if(L==1)return 0.25+A; else return 0.25-A;
    }
    if(k==4){
        if(L==1||L==3){
            B=(rho(w[0]+w[1],w[2],w[3])+rho(w[0],w[1]+w[2],w[3])+rho(w[0],w[1],w[2]+w[3]))/(4*M_PI);
            if(L==1)return 0.125+B; else return 0.375-B;
        }
        C=(rho(w[0],w[1],w[2])+rho(w[1],w[2],w[3]))/(4*M_PI);
        if(L==2)return 0.375+C; else return 0.125-C;
    }
    return NAN;
};

//Level Probability with equal weights by Barlowo et al.
double** MakeListPEW(int k){
    /*重みが全て等しいときの階層確率P(L,l) (1≤L≤l≤k)の(k+1)×(k+1)2次元配列Pを出力*/
    /* P[L][l]=P(L,l) (1≤L≤l≤k)*/
    double **P;
    int l,L;
    //2次元配列Pの確保と初期化
    P = malloc((k+1)*sizeof(double *));
    for(l=0;l<=k;l++){
        P[l] = malloc((k+1)*sizeof(double));
    };
    for(L=0;L<=k;L++){
        for(l=0;l<=k;l++)P[L][l]=NAN;
    };
    //配列有効部分の計算
    P[1][1]=1;
    for(l = 2;l <= k;l++){
        for(L = 2; L <= l-1;L++){
            P[L][1] = (P[L-1][l-1] + (l-1)*P[L][l-1])/l;
        };
        P[1][l] = 1.0/l;
        P[l][l] = 1.0/tgamma(l+1.0);/*階乗計算l!=Γ(l+1)*/
    };
    return P;
};

//表関数Q生成関数と関連関数
unsigned long encode(int m,int *ss)//長さm+1の昇順整数列ssのコード化
{
    unsigned long s,b;
    int i,p;
    s=0;
    b=1;
    p=1;
    for(i=0;i<=m;i++){
        b=b<<(ss[i]-p);
        s=s|b;
        p=ss[i];
    }
    return s;
}
int decode(unsigned long s,int *ss)//コードsを配列ssにデコード，配列長-1を出力
{
    int m,i;
    unsigned long b=1;
    m=0;//m = number of nonzero bits
    for(i=1;i<=32;i++){
        if(!((s&b)==0)){ss[m]=i;m++;}
        b=b<<1;
    }
    m--;//m = (length of ss)-1
    return m;
}
double** matrix(int m,int n)//m×n行列確保
{
    double **A;
    int i;
    A=malloc(m*sizeof(double *));
    if(A==NULL)return NULL;
    for(i=0;i<m;i++){
        A[i]=malloc(n*sizeof(double));
        if(A[i]==NULL)return NULL;
    }
    return A;
}
double* vector(int n)//n次元ベクトル確保
{
    double *v;
    v=malloc(n*sizeof(double));
    if(v==NULL)return NULL;
    return v;
}
double norm1(int n,double *v)//n次元ベクトルの1ノルム
{
    double nm;
    int i;
    nm=fabs(v[0]);
    for(i=1;i<n;i++)nm+=fabs(v[i]);
    return nm;
}
double norm8(int n,double *v)//n次元ベクトルの∞ノルム
{
    double nm;
    int i;
    nm=fabs(v[0]);
    for(i=1;i<n;i++)if(nm<fabs(v[i]))nm=fabs(v[i]);
    return nm;
}
double min(int n,double *v)//n次元ベクトルの最小要素
{
    double mn;
    int i;
    mn=v[0];
    for(i=1;i<n;i++)if(mn>v[i])mn=v[i];
    return mn;
}
int RecursionF(int N,double **S,double **f0,double sg,double sgmax,double *phi,double **f1)
{
    double feps;
    int n,i,j,imin,imax,jmin,jmax;
    feps=norm8(N,&f0[0][0])*pow(2,-53);//関数値絶対値の閾値
    if(feps==0) //then f1=0
    {
        for(i=0;i<N;i++)f1[0][i]=0;
        return 0;
    }
    else        //else f1=phi*(S.f0)
    {
        //trim f0
        jmin=0;
        while(fabs(f0[0][jmin])<feps)jmin++;
        jmax=N-1;
        while(fabs(f0[0][jmax])<feps)jmax--;
        //trim phi
        n=N/2;
        imin=n-(int)(n*sg/sgmax);
        imax=n+(int)(n*sg/sgmax);
        for(i=0;i<N;i++)f1[0][i]=0;
        for(i=imin;i<=imax;i++)
        {
            for(j=jmin;j<=jmax;j++)f1[0][i]+=S[i][j]*f0[0][j];//f1=S.f0
            f1[0][i]*=phi[i];//f1=phi*(S.f0)
        }
        return 0;
    }
};
double* MakeTableQ(int k,double *w){/*表関数Qの作成*/
    double *Q,wn[k];
    double *x,**S,**f,*phiv,*uv;
    double h=0.8,sgmax,umm,sg,sum;
    int p[k+1];
    int n,N,i,j,m,nQ;
    //準備
    nQ=pow(2,k+1);
    Q=vector(nQ);
    for(i=0;i<nQ;i++)Q[i]=INFINITY;
    for(i=0;i<k;i++)wn[i]=w[i]/norm1(k,w);//正規化重みベクトルwn
    sgmax=sqrt(1/min(k,wn));//最小標準偏差
    n=ceil(8*sgmax/h);//台半幅分割数
    N=2*n+1;
    x=vector(N);//標本点列x
    for(i=0;i<N;i++)x[i]=(i-n)*h;
    S=matrix(N,N);//積分行列
    for(i=0;i<N;i++){
        for(j=0;j<N;j++){
            S[i][j]=Snc(h,x[j],x[i]);
        }
    }
    f=matrix(k+1,N);//f[0]〜f[k]のsinc係数(標本ベクトル)
    uv=vector(k);//weight vector
    phiv=vector(N);//正規分布密度関数の標本ベクトル
    //計算開始
    p[0]=1;m=0;//初期経路pと経路高m
    while(!(m==0&&p[0]==k+1))
    {
        //(1) 不定積分
        if(m>=1&&(k+1-p[m]+m)>4)
        {
            if(m==1)
            {
                umm=0;
                for(j=p[m-1];j<p[m];j++)umm+=wn[j-1];
                sg=1/sqrt(umm);//終端σ
                for(i=0;i<N;i++)f[m-1][i]=phiPDF(sg,x[i]);//f[0]の設定
            }
            else
            {
                umm=0;
                for(j=p[m-1];j<p[m];j++)umm+=wn[j-1];
                sg=1/sqrt(umm);//終端σ
                for(i=0;i<N;i++)phiv[i]=phiPDF(sg,x[i]);
                RecursionF(N,&S[0],&f[m-2],sg,sgmax,phiv,&f[m-1]);//f[m-1]の計算(漸化式)
            }
        }
        //(2) 定積分
        if(m>=1)
        {
            if(1<=m&&m<=4)  //then Robertson公式
            {
                for(i=0;i<m;i++)
                {
                    uv[i]=0;
                    for(j=p[i];j<p[i+1];j++)uv[i]+=wn[j-1];
                }
                Q[encode(m,p)]=LvPrLe4(m,m,uv);
            }
            else            //else f[m-1]を台形則積分
            {
                sum=0;
                for(i=0;i<N;i++)sum+=f[m-1][i];
                Q[encode(m,p)]=h*sum;
            }
        }
        //(3) 経路pの更新
        if(p[m]<k+1)  //then パスを伸ばす
        {
            p[m+1]=p[m]+1;m++;
        }
        else        //else 転進
        {
            p[m-1]=p[m-1]+1;m--;
        }
    }
    //配列の解放
    free(S);free(x);free(f);free(uv);free(phiv);
    return Q;
};

//表関数Q生成関数と関連関数
unsigned long pow2(int n)// = 2^n for 0≤n<32
{
    return ((unsigned long)1)<<n;
};
double*** MakeTableP(int k,double *w)//make table P with 1≤k≤31 and k dimentional weight vector w
{
    double ***P;//P[L][m][s]=P[L,m,s]
    double *Q;//pointer of the table Q
    int m,s,k4,L,i,sv[k+1];
    unsigned long code,code0,code1,dc;
    double pr;
    //"Called" message
    // make table Q
    Q=MakeTableQ(k,w);
    // allocate P
    P=malloc(k*sizeof(double**));//index L=1,2,...,k
    P=P-1;
    for(L=1;L<=k;L++)
    {
        P[L]=malloc((k-L+1)*sizeof(double*));//index m=L,L+1,...,k
        P[L]=P[L]-L;
    }
    for(L=1;L<=k;L++)
    {
        for(m=L;m<=k;m++)
        {
            P[L][m]=malloc((k-m+1)*sizeof(double));//index s=1,2,...,k-m+1
            P[L][m]=P[L][m]-1;
        }
    }
    // calculate P[L][m][s]=P[L,m,s]
    //(1) case of m≤4 :  by Robertson et. al.
    if(k<=4)k4=k; else k4=4;//k4=min{k,4}
    for(m=1;m<=k4;m++){
        for(s=1;s<=k-m+1;s++){
            for(L=1;L<=m;L++)
            {
                P[L][m][s]=LvPrLe4(L,m,&w[s-1]);
            }
        }
    }
    //(2) case of m≥5
    for(m=5;m<=k;m++){
        for(s=1;s<=k-m+1;s++)
        {
            dc=pow2(s);
            code0=(dc/2)*(pow2(m)+1);//     code0=binary(0,..,0,1,0,..,0,1)
            code1=(dc/2)*(pow2(m+1)-1);//   code1=binary(0,..,0,1,1,..,1,1) with (s-1) zeros and (m+1) ones.
            P[m][m][s]=Q[code1];
            for(L=2;L<m;L++)P[L][m][s]=0;//zero reset
            for(code=code0+dc;code<code1;code+=dc)
            {
                L=decode(code,sv);
                if(L>=2)
                {
                    pr=Q[code];
                    for(i=0;i<=L-1;i++)pr*=P[1][sv[i+1]-sv[i]][sv[i]];
                    P[L][m][s]+=pr;
                }
            }
            P[1][m][s]=1;
            for(L=2;L<=m;L++)P[1][m][s]-=P[L][m][s];
        }
    }
    return P;
};
#endif /* Chap8_h */
